/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xstrutil.h 
 *       string conversion utilities 
 * -----------------------------------------------------------------*/

#ifndef STRUTIL_H_INCLUDED
#define STRUTIL_H_INCLUDED

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* the following is a little hack because math.h defines HUGE, and we 
 * do so as well! */

#ifdef HUGE
#  define OLDHUGE HUGE
#  undef HUGE
#endif

#include <math.h>

#ifdef OLDHUGE
#  undef HUGE
#  define HUGE OLDHUGE
#endif

/* get rid of stupid MSC's warning about unused functions being removed */
#ifdef _MSC_VER
# pragma warning(disable : 4505)
#endif

#ifndef ULONG_MAX
#  define ULONG_MAX 0xFFFFFFFFUL
#endif

#ifndef LONG_MAX
#  define LONG_MAX 0x7FFFFFFFL
#endif

#ifndef LONG_MIN
#  define LONG_MIN 0x80000000L
#endif

#if !defined(UNIX) && !defined (__unix)
#  ifdef strncasecmp
#    undef strncasecmp
#  endif
#  ifdef strcasecmp
#    undef strcasecmp
#  endif
#  ifdef __BORLANDC__
#    define strncasecmp strnicmp
#    define strcasecmp stricmp
#  else
#    define strncasecmp _strnicmp
#    define strcasecmp _stricmp
#  endif
#endif /* defined UNIX */


/* allow default values for C++ programs */
#ifdef __cplusplus
#  define DEFAULT_VAL(a) = (a)
#  define STR_ARG_LIST(str,ptr,base)\
                  const char *str, char **ptr = (char **)0, int base = 0
#else
#  define DEFAULT_VAL(a)
#  define STR_ARG_LIST(str,ptr,base)\
    const char *str, char **ptr, int base
#endif


/* for converting Best-Style number string to values */
/* this is 0-pointer proof */    
static struct __convbase_s {
  char * _modifier;
  int _base;
} _convbase [] = {
  {"/h", 16},
  {"\\h", 16},
  {"/b", 2},
  {"\\b",2},
  {0, 0}
};

/*---------------------------------------------------------------------------*
 * static int StrEQ (const char * s1, const char * s2)
 *
 * Purpose: compare two strings, return 1 if they're equal
 *---------------------------------------------------------------------------*/
static int StrEQ (const char * s1, const char * s2)
{
  int retval = 0;
   
  if (s2 && *s2)  /* s2 not empty? */
  {
    retval = ( (s1 != 0) && ( strcasecmp(s1, s2) == 0 ) );
  }
  else    /* s2 empty -> is s1 empty as well? */
  {
    retval = ( s1 == 0 || s1[0] == 0 );
  }

  return retval;
}

/*---------------------------------------------------------------------------*
 * static unsigned long Log2 (unsigned long inval)
 *
 * Purpose: get Log base 2 of number
 *---------------------------------------------------------------------------*/
static unsigned long Log2 (unsigned long inval)
{
  unsigned long outval;

  for (outval=0; inval >>= 1; outval++);

  return outval;
}

/*---------------------------------------------------------------------------*
 * static unsigned long Str2ULong (STR_ARG_LIST(str,ptr,base))
 *
 * Purpose: safe conversion from string to unsigned long
 *                allows \h, \b format
 *---------------------------------------------------------------------------*/
static unsigned long Str2ULong (STR_ARG_LIST(str,ptr,base))
{
  int i = 0;
  char * helpptr;
  unsigned long val;

  if (str == 0)
    return 0;
  else
  {
    while ( _convbase[i]._modifier &&
     strstr(str, _convbase[i]._modifier) == 0 )
      i++;

    if (_convbase[i]._modifier)
      base = _convbase[i]._base;
  }
  
  val = strtoul (str, &helpptr, base);
  if ( helpptr && *helpptr &&
       _convbase[i]._modifier &&
       helpptr == strstr(str, _convbase[i]._modifier) )
  {
    helpptr += (strlen(_convbase[i]._modifier)); /* ignore modifier */
  }

  if (ptr)
    *ptr = helpptr;

  return val;
}

/*---------------------------------------------------------------------------*
 * static int IsULong (STR_ARG_LIST(str, ptr, base))
 *
 * Purpose: find out if string is an unsigned long number (\h,\b support)
 *---------------------------------------------------------------------------*/
static int IsULong (STR_ARG_LIST(str, ptr, base))
{
  int retval = 1;
  char * helpptr;
  unsigned long val = Str2ULong (str, &helpptr, base);
  if (helpptr && *helpptr)
    retval = 0;
  else if (val == ULONG_MAX && errno == ERANGE) /* out of range */
    retval = 0;

  if (ptr)
    *ptr = helpptr;
  return retval;
}

/*---------------------------------------------------------------------------*
 * static long Str2Long (STR_ARG_LIST(str,ptr,base))
 *
 * Purpose: safe conversion from string to long
 *                allows \h, \b format
 *---------------------------------------------------------------------------*/
static long Str2Long (STR_ARG_LIST(str,ptr,base))
{
  int i = 0;
  char * helpptr;
  long val;

  if (str == 0)
    return 0;
  else
  {
    while ( _convbase[i]._modifier &&
     strstr(str, _convbase[i]._modifier) == 0 )
      i++;

    if (_convbase[i]._modifier)
      base = _convbase[i]._base;
  }
  
  val = strtol (str, &helpptr, base);

  if ( helpptr && *helpptr &&
       _convbase[i]._modifier &&
       helpptr == strstr(str, _convbase[i]._modifier) )
  {
    helpptr += (strlen(_convbase[i]._modifier)); /* ignore modifier */
  }

  if (ptr)
    *ptr = helpptr;

  return val;
}

/*---------------------------------------------------------------------------*
 * static int IsLong (STR_ARG_LIST(str, ptr, base))
 *
 * Purpose: find out if string is a long number (\h,\b support)
 *---------------------------------------------------------------------------*/
static int IsLong (STR_ARG_LIST(str, ptr, base))
{
  int retval = 1;
  char * helpptr;
  long val = Str2Long (str, &helpptr, base);
  if (helpptr && *helpptr)
    retval = 0;
  else if ( (val == LONG_MAX || val == LONG_MIN) &&  /* out of range */
     errno == ERANGE )
    retval = 0;

  if (ptr)
    *ptr = helpptr;
  return retval;
}

#ifdef __cplusplus

/*---------------------------------------------------------------------------*
 * static void Str2Mask ( const char * str,
 *
 * Purpose: convert 01x string to mask/value pair
 *---------------------------------------------------------------------------*/
static void Str2Mask ( const char * str,
         unsigned long & value,
         unsigned long & mask )
{
  int steps;
  char * helpstr;
  if (helpstr = strstr (str, "\\b")) /* okay, we're binary... */
    steps = 1;   /* 1 bin value per digit */
  else if (helpstr = strstr (str, "\\h"))
    steps = 4;   /* 4 bin values per digit */
  else
    steps = 0;

  value = 0;
  mask = 0;
  
  while (str < helpstr)
  {
    value <<= steps;
    mask <<= steps;
    if ( (*str == 'x') || (*str == 'X') )
    {
      mask |= ((1<<steps) -1);
    }
    else if ( (*str >= 'a') && (*str <= 'f') ) /* lower hex digit */
    {
      value |= (*str - 'a' + 0xa);
    }
    else if ( (*str >= 'A') && (*str <= 'F') ) /* capitalized hex digit */
    {
      value |= (*str - 'A' + 0xa);
    }
    else   /* decimal digit */
    {
      value |= (*str - '0');
    }
    str++;   /* position at next digit... */
  }
  
}

#endif /* defined  __cplusplus */

/*---------------------------------------------------------------------------*
 * static void PrintBase (char * str,
                         unsigned long val, int base, int width)
 *
 * Purpose: convert number to 'base' and print (e.g. 17 -> 11\h)
 *---------------------------------------------------------------------------*/
static void PrintBase (char * str, unsigned long val, int base, int width)
{
  char fmtstr[10];
  char helpstr[33];
  char * strptr = &helpstr[32];
  *(strptr--) = '\0';

  if (width > 0)
  {
    sprintf (fmtstr, "%%0%dl", width);
  }
  else
  {
    strcpy (fmtstr, "%l");
  }

  switch (base)
  {
   case 10:
    strcat (fmtstr, "u");
    sprintf(str, fmtstr, val);
    break;
   case 16:
    strcat (fmtstr, "x");
    sprintf(str, fmtstr, val);
    break;
   default:
    while (val)
    {
      *(strptr--) = (char) ('0' + val%base);
      val /= base;
      width--;
    }

    while (width > 0 && strptr >= helpstr)
    {
      width--;
      *(strptr--) = '0';
    }
    
    strcpy (str, strptr+1);
  }
    
       
}

/*---------------------------------------------------------------------------*
 * static void Mask2Str ( unsigned long value,
 *
 * Purpose: convert mask/value pair to 01x string
 *---------------------------------------------------------------------------*/
static void Mask2Str ( unsigned long value,
         unsigned long mask,
         char * str,
         int forceBin DEFAULT_VAL(1) )
{
  char helpstr[40];
  int isHex;
  int i, base, width;

  if (forceBin)
  {
    isHex = 0;
  }
  else
  {
    isHex = 1;
    for (i=0; i<32; i+=4)
    {
      if ( (1<<0xF & mask) != (1<<0xF) )
      {
        isHex = 0;
        break;
      }
    }
  }

  base = isHex ? 16 : 2;
  width = isHex ? 8 : 32;

  PrintBase(str, value, base, width);
  PrintBase(helpstr, mask, base, width);

  for (i=0; (size_t)i<strlen(str); i++)
    if (helpstr[i] != '0')
      str[i] = 'x';

  strcat (str, (isHex ? "\\h" : "\\b"));
}

/*****************************************************************************
 * unit conversion/display helper functions
 *****************************************************************************/

/*----------------------------------------------------------------------
 * some helper structs/variables
 *----------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*
 * struct unitstype
 * holds data for conversion:
 * each entry contains a number symbol and its maximum value for printing
 *---------------------------------------------------------------------------*/
struct unitstype{
  char * unit;
  double divisor;
};

static struct unitstype timeunits [] = {
#if defined(WIN32) || defined(WIN64)
  {"s", 1000.0},       
#else
  {"us", 1000.0},       
#endif
  {"ms", 1000.0},
  {"s",    60.0},
  {"min",   60.0}, 
  {"h",    24.0}, 
  {"d",   365.0},
  {"a",   100.0},
  {"centuries",   10.0},
  {"milleniums",   0.0},
  {0,     0.0}
};

static struct unitstype dataunits [] = {
  {"bytes", 1024.0},
  {"Kbyte", 1024.0},
  {"Mbyte", 1024.0},
  {"GByte", 1024.0},
  {"TeraByte",    0.0},
  {0,     0.0}
};

static struct unitstype frequencyunits [] = {
  {"Hz", 1000.0},
  {"kHz", 1000.0},
  {"MHz", 1000.0},
  {"GHz", 1000.0},
  {"THz",    0.0},
  {0,     0.0}
};

static struct unitstype doubleunits [] = {
  {"",   1.0},
  {0,   0.0} };

/*---------------------------------------------------------------------------*
 * static void PrintUnits (char buf[], double number, struct unitstype units[])
 *
 * Purpose: print number 'readable' with appropriate units
 * inputs: buf, must be large enough to hold result!
 *---------------------------------------------------------------------------*/
static void PrintUnits (char buf[], double number, struct unitstype units[])
{
  int i = 0;
  
  while ( (units[i+1].unit) && (fabs(number) >= units[i].divisor) )
  {
    number /= units[i].divisor;
    i++;
  }
  sprintf (buf, "%.5lg %s", number, units[i].unit);
}

/*---------------------------------------------------------------------------*
 * PrintTime, PrintData, PrintFrequency, PrintDouble:
 * all defines to PrintUnits with according data!
 *---------------------------------------------------------------------------*/
#define PrintTime(buf,a) PrintUnits(buf,(a) * 1000000.0, timeunits)
#define PrintData(buf,a) PrintUnits(buf,(a), dataunits)
#define PrintFrequency(buf,a) PrintUnits(buf,(a), frequencyunits)
#define PrintDouble(buf,a) PrintUnits(buf,(a), doubleunits)

/* don't add anything below this point */
#endif
